
        // Admin CRM JavaScript
        window.AdminCRM = {
            currentUser: null,
            chatSessions: [],
            selectedSession: null,
            messages: [],
            unsubscribes: [],
            selectedChats: new Set(),
            bulkSelectMode: false,
            isDeleting: false,
            importantChats: new Set(),
            activeTab: 'all', // 'all' or 'important'
            currentPage: 'chat-sessions', // 'chat-sessions' or 'ai-training'

            // Initialize
            init: function () {
                this.loadImportantChats();
                this.setupEventListeners();
                this.setupScrollBehavior();
                this.checkAuthState();
                this.initializeAITraining();
            },

            // Load important chats from localStorage
            loadImportantChats: function() {
                try {
                    const savedImportant = localStorage.getItem('importantChats');
                    if (savedImportant) {
                        this.importantChats = new Set(JSON.parse(savedImportant));
      //                  console.log('Loaded important chats:', this.importantChats);
                    }
                } catch (error) {
                    console.error('Error loading important chats:', error);
                    this.importantChats = new Set();
                }
            },

            // Save important chats to localStorage
            saveImportantChats: function() {
                try {
                    localStorage.setItem('importantChats', JSON.stringify([...this.importantChats]));
                } catch (error) {
                    console.error('Error saving important chats:', error);
                }
            },

            // Setup PYX-like scroll behavior
            setupScrollBehavior: function() {
                const sidebarScroll = document.getElementById('sessions-list');
                const chatMessagesContainer = document.querySelector('.chat-messages-container');

//                console.log('Setting up PYX-like scroll behavior for:', { sidebarScroll, chatMessagesContainer });

                // Each area scrolls independently - no special handling needed
                // The CSS layout handles the separation

                // Auto-scroll to bottom when new messages arrive
                this.autoScrollToBottom = function() {
                    if (chatMessagesContainer) {
                        setTimeout(() => {
                            chatMessagesContainer.scrollTop = chatMessagesContainer.scrollHeight;
                        }, 100);
                    }
                };

                // Smooth scroll to bottom function
                this.scrollToBottom = function(smooth = true) {
                    if (chatMessagesContainer) {
                        if (smooth) {
                            chatMessagesContainer.scrollTo({
                                top: chatMessagesContainer.scrollHeight,
                                behavior: 'smooth'
                            });
                        } else {
                            chatMessagesContainer.scrollTop = chatMessagesContainer.scrollHeight;
                        }
                    }
                };

                // Check if user is near bottom of chat
                this.isNearBottom = function() {
                    if (!chatMessagesContainer) return false;
                    const threshold = 100; // pixels from bottom
                    return chatMessagesContainer.scrollHeight - chatMessagesContainer.scrollTop - chatMessagesContainer.clientHeight < threshold;
                };

                // Add scroll event listener to chat messages
                if (chatMessagesContainer) {
                    chatMessagesContainer.addEventListener('scroll', () => {
                        // You can add scroll-based features here like "scroll to bottom" button
                        const isAtBottom = this.isNearBottom();
                        // Could show/hide a "scroll to bottom" button based on isAtBottom
                    });
                }
            },

            // Setup event listeners
            setupEventListeners: function () {
                // Login form
                document.getElementById('login-form').addEventListener('submit', (e) => this.handleLogin(e));
                document.getElementById('toggle-password').addEventListener('click', () => this.togglePassword());

                // Logout
                document.getElementById('logout-button').addEventListener('click', () => this.handleLogout());

                // Bulk select
                document.getElementById('bulk-select-button').addEventListener('click', () => this.toggleBulkSelect());
                document.getElementById('select-all-button').addEventListener('click', () => this.selectAllChats());
                document.getElementById('deselect-all-button').addEventListener('click', () => this.deselectAllChats());
                document.getElementById('bulk-delete-button').addEventListener('click', () => this.bulkDeleteSessions());

                // Tabs
                document.getElementById('all-chats-tab').addEventListener('click', () => this.setActiveTab('all'));
                document.getElementById('important-chats-tab').addEventListener('click', () => this.setActiveTab('important'));

                // Search and filter
                document.getElementById('search-input').addEventListener('input', (e) => this.filterSessions());
                document.getElementById('refresh-button').addEventListener('click', () => this.loadSessions());

                // Message input
                document.getElementById('admin-message-input').addEventListener('keypress', (e) => this.handleKeyPress(e));
                document.getElementById('send-admin-message').addEventListener('click', () => this.sendAdminMessage());

                // Navigation tabs
                document.getElementById('chat-sessions-tab').addEventListener('click', () => this.showChatSessions());
                document.getElementById('ai-training-tab').addEventListener('click', () => this.showAITraining());
                document.getElementById('usage-stats-tab').addEventListener('click', () => this.showUsageStats());
                document.getElementById('account-tab').addEventListener('click', () => this.showAccountSettings());
                document.getElementById('knowledgebase-tab').addEventListener('click', () => this.showKnowledgeBase());
                // Account forms
              //  document.getElementById('change-email-form').addEventListener('submit', (e) => this.handleChangeEmail(e));
                document.getElementById('change-password-form').addEventListener('submit', (e) => this.handleChangePasswordAuth(e));

                // Usage Analytics filters and refresh
                document.getElementById('filter-7-days').addEventListener('click', () => this.filterUsageStats(7));
                document.getElementById('filter-30-days').addEventListener('click', () => this.filterUsageStats(30));
                document.getElementById('filter-365-days').addEventListener('click', () => this.filterUsageStats(365));
                document.getElementById('refresh-stats').addEventListener('click', () => this.refreshUsageStats());
            },

            // Check authentication state using Firebase Auth
            checkAuthState: function () {
                if (!window.firebaseAuth || !window.firebaseOnAuthStateChanged) {
                    console.error('Firebase Auth not initialized');
                    this.showLogin();
                    return;
                }
                // Subscribe once; store unsubscribe to clean up on logout
                const unsubscribe = window.firebaseOnAuthStateChanged(window.firebaseAuth, async (user) => {
                    if (user) {
                        // Check if user authenticated with email/password provider
                        const isEmailPasswordUser = user.providerData.some(provider => provider.providerId === 'password');
                        
                        if (isEmailPasswordUser) {
                            this.currentUser = { email: user.email, uid: user.uid };
                            this.showDashboard();
                            this.loadSessions();
                            this.loadCurrentAccountInfo();
                        } else {
                            // User authenticated with a different provider, sign them out
                            try {
                                if (window.firebaseSignOut) {
                                    await window.firebaseSignOut(window.firebaseAuth);
                                }
                                this.currentUser = null;
                                this.showError('Access denied. Please log in with email and password.');
                                this.showLogin();
                            } catch (error) {
                                console.error('Error signing out user:', error);
                                this.showError('Authentication error. Please try again.');
                                this.showLogin();
                            }
                        }
                    } else {
                        this.currentUser = null;
                        this.showLogin();
                    }
                });
                this.unsubscribes.push(unsubscribe);
            },

            // Handle login
            handleLogin: async function (e) {
                e.preventDefault();
                const email = document.getElementById('email').value;
                const password = document.getElementById('password').value;

                console.log('Login attempt:', email);
                console.log('Firebase Auth available:', !!window.firebaseAuth);

                this.setLoading(true);
                this.hideError();

                try {
                    if (!window.firebaseAuth) {
                        throw new Error('Firebase Authentication not properly initialized. Please refresh the page.');
                    }

                    // Sign in with Firebase Authentication - only authorized users in Firebase Console can login
                    const userCredential = await window.firebaseSignInWithEmailAndPassword(window.firebaseAuth, email, password);
                    const user = userCredential.user;

                    // Set current user and show dashboard
                    this.currentUser = { email: user.email, uid: user.uid };
                    this.showDashboard();
                    this.loadSessions();

                    console.log('Login successful');
                } catch (error) {
                    console.error('Login error:', error);
                    let errorMessage = 'Invalid email or password. Please try again.';
                    if (error.code === 'auth/user-not-found') {
                        errorMessage = 'No account found with this email address.';
                    } else if (error.code === 'auth/wrong-password') {
                        errorMessage = 'Incorrect password. Please try again.';
                    } else if (error.code === 'auth/invalid-email') {
                        errorMessage = 'Invalid email address format.';
                    } else if (error.code === 'auth/too-many-requests') {
                        errorMessage = 'Too many failed login attempts. Please try again later.';
                    }
                    this.showError(errorMessage);
                } finally {
                    this.setLoading(false);
                }
            },



            // Handle logout
            handleLogout: async function () {
                try {
                    // Clean up listeners
                    this.unsubscribes.forEach(unsubscribe => unsubscribe());
                    this.unsubscribes = [];

                    // Sign out from Firebase Auth
                    if (window.firebaseAuth && window.firebaseSignOut) {
                        await window.firebaseSignOut(window.firebaseAuth);
                    }

                    // Clear localStorage data
                    localStorage.removeItem('adminLoggedIn');
                    localStorage.removeItem('adminEmail');
                    localStorage.removeItem('adminUID');

                    // Reset current user; UI will switch via onAuthStateChanged
                    this.currentUser = null;

                    console.log('Logout successful');
                } catch (error) {
                    console.error('Logout error:', error);
                }
            },

            // Toggle password visibility
            togglePassword: function () {
                const passwordInput = document.getElementById('password');
                const eyeClosed = document.getElementById('eye-closed');
                const eyeOpen = document.getElementById('eye-open');

                if (passwordInput.type === 'password') {
                    passwordInput.type = 'text';
                    eyeClosed.classList.add('hidden');
                    eyeOpen.classList.remove('hidden');
                } else {
                    passwordInput.type = 'password';
                    eyeClosed.classList.remove('hidden');
                    eyeOpen.classList.add('hidden');
                }
            },

            // Show/hide screens
            showLogin: function () {
                document.getElementById('login-screen').classList.remove('hidden');
                document.getElementById('crm-dashboard').classList.add('hidden');
            },

            showDashboard: function () {
                document.getElementById('login-screen').classList.add('hidden');
                document.getElementById('crm-dashboard').classList.remove('hidden');

                // Ensure stats are updated when dashboard is shown
                setTimeout(() => {
                    this.updateStats();
                }, 100);
            },

            // Loading and error states
            setLoading: function (loading) {
                const button = document.getElementById('login-button');
                const text = document.getElementById('login-text');
                const loadingDiv = document.getElementById('login-loading');

                if (loading) {
                    button.disabled = true;
                    text.classList.add('hidden');
                    loadingDiv.classList.remove('hidden');
                } else {
                    button.disabled = false;
                    text.classList.remove('hidden');
                    loadingDiv.classList.add('hidden');
                }
            },

            showError: function (message) {
                const errorDiv = document.getElementById('error-message');
                const errorText = document.getElementById('error-text');
                errorText.textContent = message;
                errorDiv.classList.remove('hidden');
            },

            hideError: function () {
                document.getElementById('error-message').classList.add('hidden');
            },

            // Load chat sessions
            loadSessions: function () {
                const sessionsRef = window.firebaseRef(window.firebaseDatabase, 'chatSessions');
                const sessionsQuery = window.firebaseQuery(sessionsRef, window.firebaseOrderByChild('lastActivity'));

                // Clean up previous listener
                this.unsubscribes.forEach(unsubscribe => unsubscribe());
                this.unsubscribes = [];

                const unsubscribe = window.firebaseOnValue(sessionsQuery, (snapshot) => {
                    const sessions = [];
                    if (snapshot.exists()) {
                        snapshot.forEach((childSnapshot) => {
                            sessions.push({
                                sessionId: childSnapshot.key,
                                ...childSnapshot.val()
                            });
                        });
                    }

                    // Sort by last activity (most recent first)
                    sessions.sort((a, b) => b.lastActivity - a.lastActivity);
                    this.chatSessions = sessions;
                    this.renderSessions();
                    this.updateStats();
                });

                this.unsubscribes.push(() => unsubscribe);
            },

            // Render sessions list
            renderSessions: function () {
                const container = document.getElementById('sessions-container');
                const loading = document.getElementById('sessions-loading');
                const empty = document.getElementById('sessions-empty');

                loading.classList.add('hidden');

                if (this.chatSessions.length === 0) {
                    empty.classList.remove('hidden');
                    container.innerHTML = '';
                    return;
                }

                empty.classList.add('hidden');

                const filteredSessions = this.getFilteredSessions();

                container.innerHTML = filteredSessions.map(session => {
                    const isSelected = this.selectedChats.has(session.sessionId);
                    const lastMessage = this.getLastMessage(session);

                    const isImportant = this.importantChats.has(session.sessionId);

                    return `
                    <div class="session-item group relative hover:bg-gray-50 transition-colors cursor-pointer ${this.selectedSession?.sessionId === session.sessionId ? 'bg-green-50' : ''
                        }" data-session-id="${session.sessionId}">

                        <div class="flex items-center px-4 py-3">
                            ${this.bulkSelectMode ? `
                                <button class="bulk-select-checkbox mr-3 text-gray-400 hover:text-blue-600" data-session-id="${session.sessionId}">
                                    ${isSelected ?
                                '<svg width="18" height="18" class="text-blue-600" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="9,11 12,14 22,4"></polyline><path d="M21,12v7a2,2 0,0 1,-2,2H5a2,2 0,0 1,-2,-2V5a2,2 0,0 1,2,-2h11"></path></svg>' :
                                '<svg width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><rect x="3" y="3" width="18" height="18" rx="2" ry="2"></rect></svg>'
                            }
                                </button>
                            ` : ''}

                            <!-- Profile Picture -->
                            <div class="flex-shrink-0 mr-3">
                                <div class="w-12 h-12 rounded-full bg-gradient-to-br from-blue-400 to-purple-500 flex items-center justify-center text-white">
                                    <svg width="24" height="24" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M20 21v-2a4 4 0 0 0-4-4H8a4 4 0 0 0-4 4v2"></path>
                                        <circle cx="12" cy="7" r="4"></circle>
                                    </svg>
                                </div>
                            </div>

                            <!-- Chat Content -->
                            <div class="flex-1 min-w-0 session-content">
                                <div class="flex items-center justify-between mb-1">
                                    <div class="flex items-center">
                                        <h3 class="text-sm font-medium text-gray-900 truncate">
                                            ${this.getDisplayName(session)}
                                        </h3>
                                        ${isImportant ? `
                                            <svg class="w-4 h-4 text-yellow-500 ml-1" fill="currentColor" viewBox="0 0 20 20">
                                                <path d="M9.049 2.927c.3-.921 1.603-.921 1.902 0l1.07 3.292a1 1 0 00.95.69h3.462c.969 0 1.371 1.24.588 1.81l-2.8 2.034a1 1 0 00-.364 1.118l1.07 3.292c.3.921-.755 1.688-1.54 1.118l-2.8-2.034a1 1 0 00-1.175 0l-2.8 2.034c-.784.57-1.838-.197-1.539-1.118l1.07-3.292a1 1 0 00-.364-1.118L2.98 8.72c-.783-.57-.38-1.81.588-1.81h3.461a1 1 0 00.951-.69l1.07-3.292z"/>
                                            </svg>
                                        ` : ''}
                                    </div>
                                    <div class="flex items-center space-x-2">
                                        <span class="text-xs text-gray-500">
                                            ${this.formatTimePYX(session.lastActivity)}
                                        </span>
                                    </div>
                                </div>
                                <div class="flex items-center justify-between">
                                    <p class="text-sm text-gray-600 truncate flex-1">
                                        ${lastMessage}
                                    </p>
                                </div>
                            </div>

                            ${!this.bulkSelectMode ? `
                                <div class="flex items-center space-x-1">
                                    <!-- Star Button -->
                                    <button class="star-session-btn p-1 rounded transition-colors opacity-0 group-hover:opacity-100 ${this.importantChats.has(session.sessionId)
                                ? 'text-yellow-500 hover:text-yellow-600'
                                : 'text-gray-400 hover:text-yellow-500 hover:bg-yellow-50'
                            }"
                                            data-session-id="${session.sessionId}"
                                            title="${this.importantChats.has(session.sessionId) ? 'Remove from Important' : 'Mark as Important'}">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="${this.importantChats.has(session.sessionId) ? 'currentColor' : 'none'}" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <polygon points="12,2 15.09,8.26 22,9.27 17,14.14 18.18,21.02 12,17.77 5.82,21.02 7,14.14 2,9.27 8.91,8.26"></polygon>
                                        </svg>
                                    </button>

                                    <!-- Delete Button -->
                                    <button class="delete-session-btn p-1 text-gray-400 hover:text-red-600 hover:bg-red-50 rounded transition-colors opacity-0 group-hover:opacity-100"
                                            data-session-id="${session.sessionId}" title="Delete Chat">
                                        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <polyline points="3,6 5,6 21,6"></polyline>
                                            <path d="M19,6v14a2,2 0,0 1,-2,2H7a2,2 0,0 1,-2,-2V6m3,0V4a2,2 0,0 1,2,-2h4a2,2 0,0 1,2,2v2"></path>
                                        </svg>
                                    </button>
                                </div>
                            ` : ''}
                        </div>
                    </div>
                `;
                }).join('');

                // Add click listeners
                const self = this;
                container.querySelectorAll('.session-content').forEach(item => {
                    item.addEventListener('click', function() {
                        const sessionId = this.closest('.session-item').dataset.sessionId;
                        const session = self.chatSessions.find(s => s.sessionId === sessionId);
                        self.selectSession(session);
                    });
                });

                // Add bulk select checkbox listeners
                container.querySelectorAll('.bulk-select-checkbox').forEach(checkbox => {
                    checkbox.addEventListener('click', function(e) {
                        e.stopPropagation();
                        const sessionId = this.dataset.sessionId;
                        self.toggleChatSelection(sessionId);
                        self.renderSessions(); // Re-render to update checkbox state
                    });
                });

                // Add star button listeners
                container.querySelectorAll('.star-session-btn').forEach(btn => {
                    btn.addEventListener('click', function(e) {
                        e.stopPropagation();
                        const sessionId = this.dataset.sessionId;
                        console.log('Star button clicked for session:', sessionId);
                        self.toggleImportant(sessionId);
                    });
                });

                // Add delete button listeners
                container.querySelectorAll('.delete-session-btn').forEach(btn => {
                    btn.addEventListener('click', function(e) {
                        e.stopPropagation();
                        const sessionId = this.dataset.sessionId;
                        self.deleteSession(sessionId);
                    });
                });
            },

            // Get filtered sessions
            getFilteredSessions: function () {
                const searchTerm = document.getElementById('search-input').value.toLowerCase();

                const filtered = this.chatSessions.filter(session => {
                    const matchesSearch = session.sessionId.toLowerCase().includes(searchTerm) ||
                        (session.userInfo?.userAgent || '').toLowerCase().includes(searchTerm);
                    const matchesTab = this.activeTab === 'all' || (this.activeTab === 'important' && this.importantChats.has(session.sessionId));
                    return matchesSearch && matchesTab;
                });

            //    console.log(`Filtering: activeTab=${this.activeTab}, importantChats size=${this.importantChats.size}, filtered count=${filtered.length}`);
                return filtered;
            },

            // Filter sessions
            filterSessions: function () {
                this.renderSessions();
            },

            // Update statistics
            updateStats: function () {
                const total = this.chatSessions.length;
                const important = this.importantChats.size;

          //      console.log(`Updating stats: total=${total}, important=${important}`);

                // Use setTimeout to ensure DOM is ready
                setTimeout(() => {
                    const allCountElement = document.getElementById('all-count');
                    const importantCountElement = document.getElementById('important-count');

        //            console.log('All count element:', allCountElement);
              //      console.log('Important count element:', importantCountElement);

                    if (allCountElement) {
                        allCountElement.textContent = total;
                        console.log('Updated all count to:', total);
                    } else {
         //               console.error('all-count element not found');
                    }

                    if (importantCountElement) {
                        importantCountElement.textContent = important;
                        console.log('Updated important count to:', important);
                    } else {
           //             console.error('important-count element not found');
                    }
                }, 0);
            },

            // Force update counters (for debugging)
            forceUpdateCounters: function() {
                console.log('Force updating counters...');
                console.log('Chat sessions:', this.chatSessions.length);
                console.log('Important chats:', this.importantChats.size);

                const allCountElement = document.getElementById('all-count');
                const importantCountElement = document.getElementById('important-count');

                if (allCountElement) {
                    allCountElement.textContent = this.chatSessions.length;
                    console.log('Force updated all count');
                } else {
                    console.error('all-count element not found during force update');
                }

                if (importantCountElement) {
                    importantCountElement.textContent = this.importantChats.size;
                    console.log('Force updated important count');
                } else {
                    console.error('important-count element not found during force update');
                }
            },

            // Select a session
            selectSession: function (session) {
                this.selectedSession = session;
                this.renderSessions(); // Re-render to show selection
                this.loadMessages(session.sessionId);
                this.showChatView(session);
            },

            // Show chat view
            showChatView: function (session) {
                document.getElementById('no-session-selected').classList.add('hidden');
                document.getElementById('chat-session-view').classList.remove('hidden');

                // Update header
                document.getElementById('session-title').textContent =
                    `Chat Session: ${session.sessionId.split('_')[2]?.substring(0, 8) || 'Unknown'}`;
                document.getElementById('session-created').textContent =
                    `Started: ${this.formatTime(session.createdAt)}`;
            },

            // Load messages for session
            loadMessages: function (sessionId) {
                const messagesRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}/messages`);
                const messagesQuery = window.firebaseQuery(messagesRef, window.firebaseOrderByChild('timestamp'));

                window.firebaseOnValue(messagesQuery, (snapshot) => {
                    const messages = [];
                    if (snapshot.exists()) {
                        snapshot.forEach((childSnapshot) => {
                            messages.push({
                                id: childSnapshot.key,
                                ...childSnapshot.val()
                            });
                        });
                    }
                    this.messages = messages;
                    this.renderMessages();
                });
            },

            // Render messages with date separators
            renderMessages: function () {
                const container = document.getElementById('messages-area');

                if (this.messages.length === 0) {
                    container.innerHTML = '';
                    return;
                }

                const messagesWithDates = this.groupMessagesByDate(this.messages);

                container.innerHTML = messagesWithDates.map(item => {
                    if (item.type === 'date') {
                        return `
                            <div class="flex justify-center my-4">
                                <div class="bg-gray-100 text-gray-600 text-xs px-3 py-1 rounded-full">
                                    ${item.label}
                                </div>
                            </div>
                        `;
                    } else {
                        const message = item.message;
                        return `
                            <div class="flex ${message.isBot ? 'justify-end' : 'justify-start'}">
                                <div class="max-w-xs lg:max-w-md px-4 py-2 rounded-lg ${message.isBot
                                ? 'text-gray-800 shadow-sm'
                                : 'bg-white text-gray-800 shadow-sm'
                            }" style="${message.isBot ? 'background-color: #E3DEEE;' : ''}">
                                    <p class="text-sm">${message.text}</p>
                                    <p class="text-xs mt-1 ${message.isBot ? 'text-gray-500' : 'text-gray-500'
                            }">
                                        ${this.formatMessageTime(message.timestamp)}
                                        ${message.isAdmin ? ' (Admin)' : ''}
                                    </p>
                                </div>
                            </div>
                        `;
                    }
                }).join('');

                // Scroll to bottom
                container.scrollTop = container.scrollHeight;
            },

            // Send admin message
            sendAdminMessage: async function () {
                const input = document.getElementById('admin-message-input');
                const message = input.value.trim();

                if (!message || !this.selectedSession) return;

                const sendButton = document.getElementById('send-admin-message');
                const sendIcon = document.getElementById('send-icon');
                const sendLoading = document.getElementById('send-loading');

                // Show loading
                sendButton.disabled = true;
                sendIcon.classList.add('hidden');
                sendLoading.classList.remove('hidden');

                try {
                    const messagesRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${this.selectedSession.sessionId}/messages`);
                    const messageRef = window.firebasePush(messagesRef);

                    const messageData = {
                        id: messageRef.key,
                        text: message,
                        isBot: true,
                        isAdmin: true,
                        adminId: this.currentUser.email,
                        timestamp: Date.now(),
                        createdAt: new Date().toISOString()
                    };

                    await window.firebaseSet(messageRef, messageData);

                    // Update last activity
                    const lastActivityRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${this.selectedSession.sessionId}/lastActivity`);
                    await window.firebaseSet(lastActivityRef, Date.now());

                    // Mark admin replied
                    const adminRepliedRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${this.selectedSession.sessionId}/adminReplied`);
                    await window.firebaseSet(adminRepliedRef, true);

                    input.value = '';
                } catch (error) {
                    console.error('Error sending admin message:', error);
                } finally {
                    sendButton.disabled = false;
                    sendIcon.classList.remove('hidden');
                    sendLoading.classList.add('hidden');
                    input.focus();
                }
            },

            // Handle key press in message input
            handleKeyPress: function (e) {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    this.sendAdminMessage();
                }
            },

            // Delete a chat session
            deleteSession: async function (sessionId) {
                if (confirm('Are you sure you want to delete this chat session? This action cannot be undone.')) {
                    try {
                        const sessionRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}`);
                        await window.firebaseRemove(sessionRef);

                        // If the deleted session was currently selected, clear the selection
                        if (this.selectedSession?.sessionId === sessionId) {
                            this.selectedSession = null;
                            this.messages = [];
                            this.renderMessages();
                            this.renderChatArea();
                        }

                        console.log(`Session ${sessionId} deleted successfully`);
                    } catch (error) {
                        console.error('Error deleting session:', error);
                        alert('Failed to delete session. Please try again.');
                    }
                }
            },

            // Important chat functions
            toggleImportant: function (sessionId) {
                console.log('Toggling important for session:', sessionId);

                if (this.importantChats.has(sessionId)) {
                    this.importantChats.delete(sessionId);
                    console.log('Removed from important');
                } else {
                    this.importantChats.add(sessionId);
                    console.log('Added to important');
                }

                // Save changes to localStorage
                this.saveImportantChats();

                // Update UI
                this.updateStats();
                this.renderSessions();
            },

            setActiveTab: function (tab) {
                this.activeTab = tab;

                // Update tab appearance - PYX style
                const allTab = document.getElementById('all-chats-tab');
                const importantTab = document.getElementById('important-chats-tab');

                // Reset both tabs to inactive state
                allTab.classList.remove('bg-green-100', 'text-green-700', 'border-green-200');
                allTab.classList.add('text-gray-600', 'hover:bg-gray-100', 'border-transparent');

                importantTab.classList.remove('bg-green-100', 'text-green-700', 'border-green-200');
                importantTab.classList.add('text-gray-600', 'hover:bg-gray-100', 'border-transparent');

                // Set active tab
                if (tab === 'all') {
                    allTab.classList.remove('text-gray-600', 'hover:bg-gray-100', 'border-transparent');
                    allTab.classList.add('bg-green-100', 'text-green-700', 'border-green-200');
                } else if (tab === 'important') {
                    importantTab.classList.remove('text-gray-600', 'hover:bg-gray-100', 'border-transparent');
                    importantTab.classList.add('bg-green-100', 'text-green-700', 'border-green-200');
                }

                this.renderSessions();

                // Update stats immediately after tab change
                this.updateStats();
            },

            // Utility functions
            formatTime: function (timestamp) {
                return new Date(timestamp).toLocaleString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    hour: 'numeric',
                    minute: '2-digit',
                    hour12: true
                });
            },

            formatMessageTime: function (timestamp) {
                return new Date(timestamp).toLocaleTimeString('en-US', {
                    hour: 'numeric',
                    minute: '2-digit',
                    hour12: true
                });
            },

            // Group messages by date with WhatsApp-style separators
            groupMessagesByDate: function (messages) {
                if (!messages || messages.length === 0) return [];

                const result = [];
                let currentDate = null;

                messages.forEach(message => {
                    const messageDate = this.getDateString(message.timestamp);

                    if (currentDate !== messageDate) {
                        currentDate = messageDate;
                        result.push({
                            type: 'date',
                            label: this.formatDateSeparator(message.timestamp)
                        });
                    }

                    result.push({
                        type: 'message',
                        message: message
                    });
                });

                return result;
            },

            // Get date string for comparison (YYYY-MM-DD)
            getDateString: function (timestamp) {
                const date = new Date(timestamp);
                return date.toISOString().split('T')[0];
            },

            // Format date separator like WhatsApp
            formatDateSeparator: function (timestamp) {
                const messageDate = new Date(timestamp);
                const today = new Date();
                const yesterday = new Date(today);
                yesterday.setDate(yesterday.getDate() - 1);

                // Reset time to compare dates only
                const messageDateOnly = new Date(messageDate.getFullYear(), messageDate.getMonth(), messageDate.getDate());
                const todayOnly = new Date(today.getFullYear(), today.getMonth(), today.getDate());
                const yesterdayOnly = new Date(yesterday.getFullYear(), yesterday.getMonth(), yesterday.getDate());

                if (messageDateOnly.getTime() === todayOnly.getTime()) {
                    return 'Today';
                } else if (messageDateOnly.getTime() === yesterdayOnly.getTime()) {
                    return 'Yesterday';
                } else {
                    return messageDate.toLocaleDateString('en-US', {
                        year: 'numeric',
                        month: 'long',
                        day: 'numeric'
                    });
                }
            },

            // Bulk select functions
            toggleBulkSelect: function () {
                this.bulkSelectMode = !this.bulkSelectMode;
                this.selectedChats.clear();

                const bulkButton = document.getElementById('bulk-select-button');
                const bulkActionsBar = document.getElementById('bulk-actions-bar');

                if (this.bulkSelectMode) {
                    bulkButton.classList.add('text-blue-600', 'bg-blue-50');
                    bulkButton.classList.remove('text-gray-500');
                    bulkActionsBar.classList.remove('hidden');
                } else {
                    bulkButton.classList.remove('text-blue-600', 'bg-blue-50');
                    bulkButton.classList.add('text-gray-500');
                    bulkActionsBar.classList.add('hidden');
                }

                this.renderSessions();
                this.updateSelectedCount();
            },

            toggleChatSelection: function (sessionId) {
                if (this.selectedChats.has(sessionId)) {
                    this.selectedChats.delete(sessionId);
                } else {
                    this.selectedChats.add(sessionId);
                }
                this.updateSelectedCount();
            },

            selectAllChats: function () {
                this.chatSessions.forEach(session => {
                    this.selectedChats.add(session.sessionId);
                });
                this.updateSelectedCount();
                this.renderSessions();
            },

            deselectAllChats: function () {
                this.selectedChats.clear();
                this.updateSelectedCount();
                this.renderSessions();
            },

            updateSelectedCount: function () {
                const count = this.selectedChats.size;
                document.getElementById('selected-count').textContent = `${count}`;
                document.getElementById('bulk-delete-button').disabled = count === 0;
            },
 

            

            // Delete functions
            deleteSession: async function (sessionId) {
                if (confirm('Are you sure you want to delete this chat session? This action cannot be undone.')) {
                    try {
                        const sessionRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}`);
                        await window.firebaseRemove(sessionRef);

                        if (this.selectedSession?.sessionId === sessionId) {
                            this.selectedSession = null;
                            this.messages = [];
                            this.renderMessages();
                            this.renderChatArea();
                        }
                    } catch (error) {
                        console.error('Error deleting session:', error);
                        alert('Failed to delete session. Please try again.');
                    }
                }
            },

            bulkDeleteSessions: async function () {
                const count = this.selectedChats.size;
                if (count === 0) return;

                if (confirm(`Are you sure you want to delete ${count} chat session(s)? This action cannot be undone.`)) {
                    this.isDeleting = true;
                    document.getElementById('delete-button-text').textContent = 'Deleting...';
                    document.getElementById('bulk-delete-button').disabled = true;

                    try {
                        const deletePromises = Array.from(this.selectedChats).map(sessionId => {
                            const sessionRef = window.firebaseRef(window.firebaseDatabase, `chatSessions/${sessionId}`);
                            return window.firebaseRemove(sessionRef);
                        });

                        await Promise.all(deletePromises);

                        // Clear selection and exit bulk mode
                        this.selectedChats.clear();
                        this.toggleBulkSelect();

                        // Clear selected session if it was deleted
                        if (this.selectedSession && this.selectedChats.has(this.selectedSession.sessionId)) {
                            this.selectedSession = null;
                            this.messages = [];
                            this.renderMessages();
                            this.renderChatArea();
                        }
                    } catch (error) {
                        console.error('Error bulk deleting sessions:', error);
                        alert('Failed to delete some sessions. Please try again.');
                    } finally {
                        this.isDeleting = false;
                        document.getElementById('delete-button-text').textContent = 'Delete';
                        this.updateSelectedCount();
                    }
                }
            },

            getSessionPreview: function (session) {
                if (!session.messages) return 'No messages yet';
                const messageKeys = Object.keys(session.messages);
                if (messageKeys.length === 0) return 'No messages yet';
                const lastMessageKey = messageKeys[messageKeys.length - 1];
                const lastMessage = session.messages[lastMessageKey];
                return lastMessage.text.length > 50 ?
                    lastMessage.text.substring(0, 50) + '...' :
                    lastMessage.text;
            },

            // PYX-style helper functions
            getDisplayName: function (session) {
                // Try to get a meaningful name from session data
                if (session.userInfo?.name) return session.userInfo.name;
                if (session.userInfo?.email) return session.userInfo.email.split('@')[0];
                const sessionId = session.sessionId.split('_')[2] || 'Unknown';
                return `User ${sessionId.substring(0, 6)}`;
            },

            getLastMessage: function (session) {
                if (!session.messages) return 'No messages yet';
                const messageKeys = Object.keys(session.messages);
                if (messageKeys.length === 0) return 'No messages yet';
                const lastMessageKey = messageKeys[messageKeys.length - 1];
                const lastMessage = session.messages[lastMessageKey];

                // Add sender prefix for clarity
                const sender = lastMessage.sender === 'user' ? '' : 'Admin: ';
                const text = lastMessage.text.length > 40 ?
                    lastMessage.text.substring(0, 40) + '...' :
                    lastMessage.text;
                return sender + text;
            },



            formatTimePYX: function (timestamp) {
                const now = new Date();
                const date = new Date(timestamp);
                const diffInHours = (now - date) / (1000 * 60 * 60);

                if (diffInHours < 24) {
                    // Same day - show time
                    return date.toLocaleTimeString('en-US', {
                        hour: 'numeric',
                        minute: '2-digit',
                        hour12: true
                    });
                } else if (diffInHours < 168) {
                    // Within a week - show day
                    return date.toLocaleDateString('en-US', { weekday: 'short' });
                } else {
                    // Older - show date
                    return date.toLocaleDateString('en-US', {
                        month: 'short',
                        day: 'numeric'
                    });
                }
            },

            // AI Training Functions
            showChatSessions: function() {
                this.currentPage = 'chat-sessions';

                // Hide AI training, account, and Usage Analytics content
                document.getElementById('ai-training-content').classList.add('hidden');
                document.getElementById('account-content').classList.add('hidden');
                document.getElementById('usage-stats-content').classList.add('hidden');
                document.getElementById('knowledgebase-content').classList.add('hidden');

                // Show chat sessions list
                document.getElementById('chat-sessions-list').classList.remove('hidden');
                document.getElementById('chat-sessions-list').classList.add('w-1/3');
                document.getElementById('chat-sessions-list').classList.remove('w-0');

                // Show appropriate chat content
                if (this.selectedSessionId) {
                    document.getElementById('chat-session-view').classList.remove('hidden');
                    document.getElementById('no-session-selected').classList.add('hidden');
                } else {
                    document.getElementById('no-session-selected').classList.remove('hidden');
                    document.getElementById('chat-session-view').classList.add('hidden');
                }

                // Update sidebar navigation styles
                document.getElementById('chat-sessions-tab').classList.add('text-white', 'bg-blue-600');
                document.getElementById('chat-sessions-tab').classList.remove('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('ai-training-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('ai-training-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('usage-stats-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('usage-stats-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('account-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('account-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('knowledgebase-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('knowledgebase-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
            },

            showAITraining: function() {
                this.currentPage = 'ai-training';

                // Hide chat-related, account, and Usage Analytics content
                document.getElementById('no-session-selected').classList.add('hidden');
                document.getElementById('chat-session-view').classList.add('hidden');
                document.getElementById('account-content').classList.add('hidden');
                document.getElementById('usage-stats-content').classList.add('hidden');
                document.getElementById('knowledgebase-content').classList.add('hidden');

                // Hide chat sessions list completely when in settings
                document.getElementById('chat-sessions-list').classList.add('hidden');
                document.getElementById('chat-sessions-list').classList.remove('w-1/3');
                document.getElementById('chat-sessions-list').classList.add('w-0');

                // Show AI training content
                document.getElementById('ai-training-content').classList.remove('hidden');

                // Update sidebar navigation styles
                document.getElementById('ai-training-tab').classList.add('text-white', 'bg-blue-600');
                document.getElementById('ai-training-tab').classList.remove('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('chat-sessions-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('chat-sessions-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('usage-stats-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('usage-stats-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('account-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('account-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('knowledgebase-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('knowledgebase-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');

                // Initialize AI training if not already done
                if (!this.aiTrainingInitialized) {
                    this.initializeAITrainingMain();
                }

                // Load current configuration
                this.loadCurrentAIConfigMain();
            },

            // Account Settings Functions
            showAccountSettings: function() {
                this.currentPage = 'account-settings';

                // Hide chat-related content
                document.getElementById('no-session-selected').classList.add('hidden');
                document.getElementById('chat-session-view').classList.add('hidden');
                document.getElementById('ai-training-content').classList.add('hidden');
                document.getElementById('usage-stats-content').classList.add('hidden');
                document.getElementById('knowledgebase-content').classList.add('hidden');

                // Hide chat sessions list completely when in account settings
                document.getElementById('chat-sessions-list').classList.add('hidden');
                document.getElementById('chat-sessions-list').classList.remove('w-1/3');
                document.getElementById('chat-sessions-list').classList.add('w-0');

                // Show account content
                document.getElementById('account-content').classList.remove('hidden');

                // Update sidebar navigation styles
                document.getElementById('account-tab').classList.add('text-white', 'bg-blue-600');
                document.getElementById('account-tab').classList.remove('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('chat-sessions-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('chat-sessions-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('ai-training-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('ai-training-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('usage-stats-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('usage-stats-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('knowledgebase-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('knowledgebase-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');

                // Load current account info
                this.loadCurrentAccountInfo();
            },

            showKnowledgeBase: function() {
                this.currentPage = 'knowledgebase';

                // Hide other content
                document.getElementById('no-session-selected').classList.add('hidden');
                document.getElementById('chat-session-view').classList.add('hidden');
                document.getElementById('ai-training-content').classList.add('hidden');
                document.getElementById('account-content').classList.add('hidden');
                document.getElementById('usage-stats-content').classList.add('hidden');

                // Hide chat sessions list completely when in knowledgebase
                document.getElementById('chat-sessions-list').classList.add('hidden');
                document.getElementById('chat-sessions-list').classList.remove('w-1/3');
                document.getElementById('chat-sessions-list').classList.add('w-0');

                // Show knowledgebase content
                document.getElementById('knowledgebase-content').classList.remove('hidden');

                // Update sidebar navigation styles
                document.getElementById('knowledgebase-tab').classList.add('text-white', 'bg-blue-600');
                document.getElementById('knowledgebase-tab').classList.remove('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('chat-sessions-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('chat-sessions-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('ai-training-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('ai-training-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('usage-stats-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('usage-stats-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('account-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('account-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');

                // Load current knowledgebase status
                this.loadknowledgebaseTab();
            },

            loadCurrentAccountInfo: function() {
                // Get email from current authenticated user
                const adminEmail = this.currentUser?.email || 'admin@example.com';

                // Update email displays
                document.getElementById('current-email').textContent = adminEmail.length > 20 ? adminEmail.substring(0, 20) + '...' : adminEmail;
                document.getElementById('display-current-email').textContent = adminEmail;

                // Update avatar letter
                const avatarLetter = adminEmail.charAt(0).toUpperCase();
                document.querySelector('#account-tab .text-white').textContent = avatarLetter;
            },

            showUsageStats: function() {
                this.currentPage = 'usage-stats';

                // Hide chat-related content
                document.getElementById('no-session-selected').classList.add('hidden');
                document.getElementById('chat-session-view').classList.add('hidden');
                document.getElementById('ai-training-content').classList.add('hidden');
                document.getElementById('account-content').classList.add('hidden');
                document.getElementById('knowledgebase-content').classList.add('hidden');

                // Hide chat sessions list completely when in Usage Analytics
                document.getElementById('chat-sessions-list').classList.add('hidden');
                document.getElementById('chat-sessions-list').classList.remove('w-1/3');
                document.getElementById('chat-sessions-list').classList.add('w-0');

                // Show Usage Analytics content
                document.getElementById('usage-stats-content').classList.remove('hidden');

                // Update sidebar navigation styles
                document.getElementById('usage-stats-tab').classList.add('text-white', 'bg-blue-600');
                document.getElementById('usage-stats-tab').classList.remove('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('chat-sessions-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('chat-sessions-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('ai-training-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('ai-training-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('account-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('account-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');
                document.getElementById('knowledgebase-tab').classList.remove('text-white', 'bg-blue-600');
                document.getElementById('knowledgebase-tab').classList.add('text-gray-600', 'hover:text-blue-600', 'hover:bg-blue-50');

                // Initialize with default filter (7 days)
                this.filterUsageStats(7);
                
                // Initialize crawl requests data - ensure it's always visible
                this.fetchCrawlRequestsStats();
                
                // Ensure crawl requests section is visible
                const crawlSection = document.getElementById('crawl-requests-section');
                if (crawlSection) {
                    crawlSection.style.display = 'block';
                }
                
                // Ensure crawl requests content is properly initialized
                const crawlContent = document.getElementById('crawl-requests-content');
                if (crawlContent) {
                    // If it's still showing loading, make sure it's visible
                    if (crawlContent.innerHTML.includes('animate-spin')) {
                        // Content is still loading, which is fine
                    }
                }
            },

            filterUsageStats: function(days) {
                // Update filter button styles
                document.getElementById('filter-7-days').classList.remove('bg-blue-600', 'text-white');
                document.getElementById('filter-7-days').classList.add('bg-gray-200', 'text-gray-700');
                document.getElementById('filter-30-days').classList.remove('bg-blue-600', 'text-white');
                document.getElementById('filter-30-days').classList.add('bg-gray-200', 'text-gray-700');
                document.getElementById('filter-365-days').classList.remove('bg-blue-600', 'text-white');
                document.getElementById('filter-365-days').classList.add('bg-gray-200', 'text-gray-700');
                
                // Highlight selected filter
                if (days === 7) {
                    document.getElementById('filter-7-days').classList.remove('bg-gray-200', 'text-gray-700');
                    document.getElementById('filter-7-days').classList.add('bg-blue-600', 'text-white');
                } else if (days === 30) {
                    document.getElementById('filter-30-days').classList.remove('bg-gray-200', 'text-gray-700');
                    document.getElementById('filter-30-days').classList.add('bg-blue-600', 'text-white');
                } else if (days === 365) {
                    document.getElementById('filter-365-days').classList.remove('bg-gray-200', 'text-gray-700');
                    document.getElementById('filter-365-days').classList.add('bg-blue-600', 'text-white');
                }
                
                // Show loading indicator
                document.getElementById('stats-loading').classList.remove('hidden');
                document.getElementById('stats-content').classList.add('hidden');
                document.getElementById('stats-error').classList.add('hidden');
                
                // Get the latest date from Firebase data or use current date as fallback
                const latestDateRef = window.firebaseRef(window.firebaseDatabase, 'messageCompletions');
                window.firebaseOnValue(latestDateRef, (snapshot) => {
                    let latestDate = new Date();
                    if (snapshot.exists()) {
                        const data = snapshot.val();
                        const dates = Object.keys(data).sort();
                        if (dates.length > 0) {
                            latestDate = new Date(dates[dates.length - 1]);
                        }
                    }
                    
                    // Calculate date range based on latest date
                    const endDate = new Date(latestDate);
                    endDate.setHours(23, 59, 59, 999); // End of the latest day
                    
                    const startDate = new Date(latestDate);
                    startDate.setDate(startDate.getDate() - days + 1); // Inclusive of latest date
                    startDate.setHours(0, 0, 0, 0); // Start of the day
                    
                    // Format dates for Firebase query
                    const startFormatted = startDate.toISOString().split('T')[0];
                    const endFormatted = endDate.toISOString().split('T')[0];
                    
                    // Fetch data from Firebase and pass the actual latest date
                    this.fetchUsageStats(startFormatted, endFormatted, latestDate.toISOString().split('T')[0]);
                    
                    // Fetch crawl requests data
                    this.fetchCrawlRequestsStats();
                }, { onlyOnce: true });
            },

            fetchUsageStats: function(startDate, endDate, latestDateFromFirebase) {
                try {
                    // Create reference to messageCompletions in Firebase
                    const completionsRef = window.firebaseRef(window.firebaseDatabase, 'messageCompletions');
                    
                    // Set up real-time listener
                    const unsubscribe = window.firebaseOnValue(completionsRef, (snapshot) => {
                        document.getElementById('stats-loading').classList.add('hidden');
                        document.getElementById('stats-content').classList.remove('hidden');
                        
                        if (snapshot.exists()) {
                            const data = snapshot.val();
                            // Filter data for the selected date range and prepare for chart
                            const filteredData = this.filterDataForDateRange(data, startDate, endDate);
                            this.renderUsageStatsChart(filteredData, startDate, endDate, latestDateFromFirebase);
                        } else {
                            // No data available
                            document.getElementById('stats-content').innerHTML = `
                                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Usage Analytics (${startDate} to ${endDate})</h3>
                                    <p class="text-gray-500">No usage data available for the selected period.</p>
                                </div>
                            `;
                        }
                        
                        // Do not update crawl requests content here to keep it independent
                    }, (error) => {
                        // Handle error
                        document.getElementById('stats-loading').classList.add('hidden');
                        document.getElementById('stats-error').classList.remove('hidden');
                        console.error('Error fetching Usage Analytics:', error);
                        
                        // Do not update crawl requests content here to keep it independent
                    });
                    
                    // Store unsubscribe function for cleanup if needed
                    this.usageStatsUnsubscribe = unsubscribe;
                } catch (error) {
                    // Handle error
                    document.getElementById('stats-loading').classList.add('hidden');
                    document.getElementById('stats-error').classList.remove('hidden');
                    console.error('Error setting up Usage Analytics listener:', error);
                }
            },

            fetchCrawlRequestsStats: function() {
                // Ensure the Usage Analytics content is loaded before trying to render
                if (!document.getElementById('usage-stats-content')) {
                    // If Usage Analytics content isn't loaded yet, wait a bit and try again
                    setTimeout(() => this.fetchCrawlRequestsStats(), 100);
                    return;
                }
                
                try {
                    // Create reference to knowledgeMonthlyUsage in Firebase
                    const crawlRequestsRef = window.firebaseRef(window.firebaseDatabase, 'knowledgeMonthlyUsage');
                    
                    // Set up real-time listener
                    const unsubscribe = window.firebaseOnValue(crawlRequestsRef, (snapshot) => {
                        // Always update the content regardless of data existence
                        if (snapshot.exists()) {
                            const data = snapshot.val();
                            this.renderCrawlRequestsChart(data);
                        } else {
                            // No data available - show empty state
                            this.renderCrawlRequestsChart({});
                        }
                    }, (error) => {
                        console.error('Error fetching crawl requests stats:', error);
                        // Do not update crawl requests content here to keep it independent
                        // Just log the error and continue showing previous data if available
                    });
                    
                    // Store unsubscribe function for cleanup if needed
                    this.crawlRequestsUnsubscribe = unsubscribe;
                } catch (error) {
                    console.error('Error setting up crawl requests stats listener:', error);
                    // Update crawl requests content to show error message
                    const crawlRequestsContent = document.getElementById('crawl-requests-content');
                    if (crawlRequestsContent) {
                        crawlRequestsContent.innerHTML = `
                            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                                <h3 class="text-lg font-semibold text-gray-900 mb-4">Knowledge Base Requests</h3>
                                <p class="text-gray-500">Error loading crawl requests data. Please try again.</p>
                            </div>
                        `;
                    }
                }
            },

            filterDataForDateRange: function(data, startDate, endDate) {
                const result = {};
                const start = new Date(startDate);
                const end = new Date(endDate);
                
                // Ensure start and end dates are at the beginning/end of the day
                start.setHours(0, 0, 0, 0);
                end.setHours(23, 59, 59, 999);
                
                // Create a map of all dates in the range
                const dateRange = {};
                const currentDate = new Date(start);
                while (currentDate <= end) {
                    const dateStr = currentDate.toISOString().split('T')[0];
                    dateRange[dateStr] = data[dateStr] || 0;
                    currentDate.setDate(currentDate.getDate() + 1);
                }
                
                // Only include dates that are within the range
                for (const dateStr in dateRange) {
                    const date = new Date(dateStr);
                    // Check if the date is within the range
                    if (date >= start && date <= end) {
                        result[dateStr] = dateRange[dateStr];
                    }
                }
                
                return result;
            },

            renderUsageStatsChart: function(data, startDate, endDate, latestDateFromFirebase) {
                // Create arrays of only the dates that have data
                const dates = Object.keys(data).sort();
                
                // Ensure the latest date from Firebase is included in the chart
                const latestDate = latestDateFromFirebase || endDate;
                
                // Prepare data for chart using only actual data points
                const values = dates.map(date => data[date]);
                
                // Get latest date's value
                const latestValue = data[latestDate] || 0;
                
                // Create enhanced SVG bar chart visualization
                let chartHtml = '';
                if (dates.length > 0) {
                    // Chart dimensions
                    const width = 800;
                    const height = 400;
                    const margin = { top: 20, right: 30, bottom: 60, left: 60 };
                    const chartWidth = width - margin.left - margin.right;
                    const chartHeight = height - margin.top - margin.bottom;
                    
                    // Calculate max value for scaling (with some padding)
                    const maxValue = Math.max(...values, 1) * 1.1;
                    
                    // Function to round numbers for y-axis labels
                    const roundToNiceNumber = (num) => {
                        if (num >= 1000) return Math.round(num / 100) * 100;
                        if (num >= 100) return Math.round(num / 10) * 10;
                        if (num >= 10) return Math.round(num);
                        return Math.ceil(num);
                    };
                    
                    // Generate y-axis ticks
                    const yTicks = [];
                    const tickCount = 5;
                    for (let i = 0; i <= tickCount; i++) {
                        const value = (maxValue * i) / tickCount;
                        yTicks.push({
                            value: value,
                            label: roundToNiceNumber(value)
                        });
                    }
                    
                    // Helper function to convert date format from YYYY-MM-DD to DD Mon YYYY
                    const formatDate = (dateStr) => {
                        const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
                        const parts = dateStr.split('-');
                        const year = parts[0];
                        const month = months[parseInt(parts[1], 10) - 1];
                        const day = parts[2];
                        return day + ' ' + month + ' ' + year;
                    };
                    
                    // Determine which dates to show on x-axis (ensure even spacing for 30-day view)
                    const rangeDays = Math.floor((new Date(endDate) - new Date(startDate)) / (1000 * 60 * 60 * 24)) + 1;
                    let dateLabels = [];
                    if (dates.length <= 10) {
                        // 10 or fewer points: show all labels
                        dateLabels = dates.map((date, index) => ({
                            date,
                            index,
                            display: formatDate(date)
                        }));
                    } else if (rangeDays >= 28 && rangeDays <= 32) {
                        // Around 30-day view: keep all dates so downstream selection picks 7 evenly across full range
                        dateLabels = dates.map((date, index) => ({
                            date,
                            index,
                            display: formatDate(date)
                        }));
                    } else {
                        // Larger ranges: subsample to ~10 labels
                        const step = Math.ceil(dates.length / 10);
                        for (let i = 0; i < dates.length; i += step) {
                            dateLabels.push({
                                date: dates[i],
                                index: i,
                                display: formatDate(dates[i])
                            });
                        }
                        // Always include the last date
                        if (dateLabels[dateLabels.length - 1].index !== dates.length - 1) {
                            dateLabels.push({
                                date: dates[dates.length - 1],
                                index: dates.length - 1,
                                display: formatDate(dates[dates.length - 1])
                            });
                        }
                    }
                    
                    // Bar dimensions - improved calculation to properly fill space
                    const totalBars = dates.length;
                    // Calculate bar width and spacing based on available space and number of bars
                    let barWidth, barPadding;
                    if (totalBars <= 30) {
                        // For 7-day and 30-day views, show all bars with consistent spacing
                        barWidth = Math.max(4, Math.min(30, chartWidth / Math.max(totalBars, 1) * 0.8));
                        barPadding = (chartWidth - (barWidth * totalBars)) / Math.max(totalBars - 1, 1);
                    } else {
                        // For 365-day view, limit bars to prevent overcrowding
                        barWidth = 2;
                        barPadding = (chartWidth - (barWidth * totalBars)) / Math.max(totalBars - 1, 1);
                    }
                    
                    // Generate bars
                    const bars = dates.map((date, index) => {
                        const value = data[date];
                        const x = margin.left + index * (barWidth + barPadding);
                        const barHeight = (value / maxValue) * chartHeight;
                        const y = margin.top + chartHeight - barHeight;
                        
                        return {
                            x: x,
                            y: y,
                            width: barWidth,
                            height: barHeight,
                            value: value,
                            date: date,
                            isLatest: date === latestDate
                        };
                    });
                    
                    chartHtml = `
                        <div class="mb-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Usage Analytics (${formatDate(startDate)} to ${formatDate(endDate)})</h3>
                            <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
                                <svg width="${width}" height="${height}" class="overflow-visible">
                                    <!-- Y-axis grid lines and labels -->
                                    ${yTicks.map(tick => `
                                        <line x1="${margin.left}" y1="${margin.top + chartHeight - (tick.value / maxValue) * chartHeight}"
                                              x2="${width - margin.right}" y2="${margin.top + chartHeight - (tick.value / maxValue) * chartHeight}"
                                              stroke="#e5e7eb" stroke-width="1" />
                                        <text x="${margin.left - 10}" y="${margin.top + chartHeight - (tick.value / maxValue) * chartHeight + 4}"
                                              font-size="12" fill="#6b7280" text-anchor="end">${tick.label}</text>
                                    `).join('')}
                                    
                                    <!-- X-axis -->
                                    <line x1="${margin.left}" y1="${margin.top + chartHeight}"
                                          x2="${width - margin.right}" y2="${margin.top + chartHeight}"
                                          stroke="#6b7280" stroke-width="1" />
                                    
                                    <!-- X-axis labels - 12 months for 365-day view, evenly spaced; 7 labels for 30-day view -->
                                    ${(() => {
                                        const total = dates.length;
                                        const msPerDay = 24 * 60 * 60 * 1000;
                                        const sDate = new Date(startDate);
                                        const eDate = new Date(endDate);
                                        const rangeDays = Math.floor((eDate - sDate) / msPerDay) + 1;

                                        if (rangeDays >= 330) {
                                            // 365-day view: show 12 months abbreviated (Jan, Feb, ...), evenly spaced across full width
                                            const monthsShort = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
                                            const labels = [];
                                            for (let i = 0; i < 12; i++) {
                                                const x = margin.left + (i * (chartWidth / 11)); // first at left edge, last at right edge
                                                labels.push({ x, display: monthsShort[i] });
                                            }
                                            return labels;
                                        }

                                        // Around 30-day view: 7 evenly spaced labels across full range
                                        if (rangeDays >= 28 && rangeDays <= 32) {
                                            const desired = 7;
                                            const step = (total - 1) / (desired - 1);
                                            const selected = [];
                                            const used = new Set();
                                            for (let i = 0; i < desired; i++) {
                                                const rawIdx = Math.round(i * step);
                                                let idx = Math.min(total - 1, Math.max(0, rawIdx));
                                                while (used.has(idx) && idx < total - 1) idx++;
                                                used.add(idx);
                                                selected.push({ index: idx, display: formatDate(dates[idx]) });
                                            }
                                            return selected;
                                        }

                                        // Default: show up to 10 evenly spaced labels
                                        const desired = Math.min(10, total);
                                        const step = (total - 1) / (desired - 1);
                                        const labels = [];
                                        const used = new Set();
                                        for (let i = 0; i < desired; i++) {
                                            const rawIdx = Math.round(i * step);
                                            let idx = Math.min(total - 1, Math.max(0, rawIdx));
                                            while (used.has(idx) && idx < total - 1) idx++;
                                            used.add(idx);
                                            labels.push({ index: idx, display: formatDate(dates[idx]) });
                                        }
                                        return labels;
                                    })().map(label => {
                                        const x = (label.x !== undefined) ? label.x : (margin.left + label.index * (barWidth + barPadding) + barWidth / 2);
                                        return `
                                            <text x="${x}" y="${margin.top + chartHeight + 22}"
                                                  font-size="11" fill="#6b7280" text-anchor="middle">${label.display}</text>
                                        `;
                                    }).join('')}
                                    
                                    <!-- Bars -->
                                    ${bars.map(bar => `
                                        <rect x="${bar.x}" y="${bar.y}" width="${bar.width}" height="${bar.height}"
                                              fill="${bar.isLatest ? '#3b82f6' : '#93c5fd'}"
                                              class="bar"
                                              data-date="${bar.date}"
                                              data-value="${bar.value}"
                                              style="cursor: pointer;">
                                            <title>${bar.date}: ${bar.value} executions</title>
                                        </rect>
                                    `).join('')}
                                    
                                    <!-- Y-axis label -->
                                    <text x="15" y="${margin.top + chartHeight / 2}"
                                          font-size="14" fill="#6b7280" text-anchor="middle"
                                          transform="rotate(-90, 15, ${margin.top + chartHeight / 2})">No. of Ai Replies</text>
                                    
                                    <!-- X-axis label -->
                                 
                                </svg>
                                
                                <!-- Tooltip -->
                                <div id="chart-tooltip" class="hidden absolute bg-gray-800 text-white text-sm px-3 py-2 rounded shadow-lg z-10"></div>
                            </div>
                           
                        </div>
                        <div class="bg-gray-50 p-4 rounded-lg">
                            <h4 class="font-medium text-gray-900 mb-2">Summary</h4>
                            <div class="grid grid-cols-3 gap-4">
                                <div class="text-center">
                                    <div class="text-2xl font-bold text-blue-600">${values.reduce((a, b) => a + b, 0)}</div>
                                    <div class="text-xs text-gray-500">Total Replies</div>
                                </div>
                                <div class="text-center">
                                    <div class="text-2xl font-bold text-green-600">${dates.length}</div>
                                    <div class="text-xs text-gray-500">Days With Activity</div>
                                </div>
                                <div class="text-center">
                                    <div class="text-2xl font-bold text-purple-600">${dates.length > 0 ? Math.round(values.reduce((a, b) => a + b, 0) / dates.length) : 0}</div>
                                    <div class="text-xs text-gray-500">Average Per Day</div>
                                </div>
                            </div>
                        </div>
                    `;
                } else {
                    chartHtml = `
                        <div class="mb-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Usage Analytics (${formatDate(startDate)} to ${formatDate(endDate)})</h3>
                          
                        </div>
                        <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                            <p class="text-gray-500">No usage data available for the selected period.</p>
                        </div>
                    `;
                }
                
                document.getElementById('stats-content').innerHTML = chartHtml;
                
                // Add interactivity to the chart after rendering
                if (dates.length > 0) {
                    const bars = document.querySelectorAll('.bar');
                    const tooltip = document.getElementById('chart-tooltip');
                    
                    bars.forEach(bar => {
                        bar.addEventListener('mouseenter', (e) => {
                            const date = e.target.getAttribute('data-date');
                            const value = e.target.getAttribute('data-value');
                            const rect = e.target.getBoundingClientRect();
                            const containerRect = e.target.closest('svg').getBoundingClientRect();
                            
                            tooltip.innerHTML = `
                                <div class="font-medium">${date}</div>
                                <div class="text-blue-300">${value} executions</div>
                            `;
                            tooltip.classList.remove('hidden');
                            tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';
                            tooltip.style.top = (rect.top - tooltip.offsetHeight - 10) + 'px';
                        });
                        
                        bar.addEventListener('mouseleave', () => {
                            tooltip.classList.add('hidden');
                        });
                    });
                }
            },

            refreshUsageStats: function() {
                // Get currently selected filter
                let days = 7; // Default to 7 days
                if (document.getElementById('filter-30-days').classList.contains('bg-blue-600')) {
                    days = 30;
                } else if (document.getElementById('filter-365-days').classList.contains('bg-blue-600')) {
                    days = 365;
                }

                // Refresh with current filter
                this.filterUsageStats(days);
                
                // Do not refresh crawl requests data here to keep it independent
            },

            renderCrawlRequestsChart: function(data) {
                // Check if the crawl requests content element exists
                const crawlRequestsContent = document.getElementById('crawl-requests-content');
                if (!crawlRequestsContent) {
                    console.warn('Crawl requests content element not found');
                    return;
                }
                
                // Create fixed 12 months for a target year and map values (show all months Jan-Dec)
                const dataKeys = Object.keys(data || {}).sort();
                const targetYear = dataKeys.length > 0 ? parseInt(dataKeys[dataKeys.length - 1].split('-')[0], 10) : new Date().getFullYear();
                const months = Array.from({ length: 12 }, (_, i) => `${targetYear}-${String(i + 1).padStart(2, '0')}`);
                const values = months.map(month => data[month] || 0);
                
                // Get latest month's value based on available data or default to December of target year
                const latestMonth = dataKeys.length > 0 ? dataKeys[dataKeys.length - 1] : `${targetYear}-12`;
                const latestValue = data[latestMonth] || 0;
                
                // Create enhanced SVG bar chart visualization
                let chartHtml = '';
                if (months.length > 0 || Object.keys(data).length > 0) {
                    // Chart dimensions
                    const width = 800;
                    const height = 400;
                    const margin = { top: 20, right: 30, bottom: 110, left: 60 };
                    const chartWidth = width - margin.left - margin.right;
                    const chartHeight = height - margin.top - margin.bottom;
                    
                    // Calculate max value for scaling (with some padding)
                    const maxValue = Math.max(...values, 1) * 1.1;
                    
                    // Function to round numbers for y-axis labels
                    const roundToNiceNumber = (num) => {
                        if (num >= 1000) return Math.round(num / 100) * 100;
                        if (num >= 100) return Math.round(num / 10) * 10;
                        if (num >= 10) return Math.round(num);
                        return Math.ceil(num);
                    };
                    
                    // Generate y-axis ticks
                    const yTicks = [];
                    const tickCount = 5;
                    for (let i = 0; i <= tickCount; i++) {
                        const value = (maxValue * i) / tickCount;
                        yTicks.push({
                            value: value,
                            label: roundToNiceNumber(value)
                        });
                    }
                    
                    // Helper function to format month names (short names)
                    const formatMonth = (monthStr) => {
                        const monthsShort = ['Jan','Feb','Mar','Apr','May','Jun','Jul','Aug','Sep','Oct','Nov','Dec'];
                        const parts = monthStr.split('-');
                        if (parts.length === 2) {
                            const monthIndex = parseInt(parts[1], 10) - 1;
                            return monthsShort[monthIndex] || monthStr;
                        }
                        return monthStr;
                    };
                    
                    // Determine which months to show on x-axis (show up to 12 months)
                    let monthLabels = [];
                    if (months.length <= 12) {
                        monthLabels = months.map((month, index) => ({
                            month: month,
                            index: index,
                            display: formatMonth(month)
                        }));
                    } else {
                        // If more than 12 months are present, pick 12 evenly spaced labels
                        const desired = 12;
                        const step = Math.ceil((months.length - 1) / (desired - 1));
                        for (let i = 0; i < months.length; i += step) {
                            monthLabels.push({
                                month: months[i],
                                index: i,
                                display: formatMonth(months[i])
                            });
                            if (monthLabels.length === desired) break;
                        }
                        // Ensure last month is included
                        if (monthLabels[monthLabels.length - 1]?.index !== months.length - 1) {
                            monthLabels.push({
                                month: months[months.length - 1],
                                index: months.length - 1,
                                display: formatMonth(months[months.length - 1])
                            });
                        }
                        // Cap at 12 labels
                        if (monthLabels.length > desired) monthLabels = monthLabels.slice(0, desired);
                    }
                    
                    // Bar dimensions - improved calculation to properly fill space
                    const totalBars = Math.max(months.length, 1);
                    // Calculate bar width and spacing based on available space and number of bars
                    const barWidth = Math.max(10, Math.min(40, chartWidth / Math.max(totalBars, 1) * 0.8));
                    const barPadding = (chartWidth - (barWidth * totalBars)) / Math.max(totalBars - 1, 1);
                    
                    // Generate bars
                    const bars = months.map((month, index) => {
                        const value = data[month] || 0;
                        const x = margin.left + index * (barWidth + barPadding) + barPadding / 2;
                        const barHeight = (value / maxValue) * chartHeight;
                        const y = margin.top + chartHeight - barHeight;
                        
                        return {
                            x: x,
                            y: y,
                            width: barWidth,
                            height: barHeight,
                            value: value,
                            month: month,
                            isLatest: month === latestMonth
                        };
                    });
                    
                    chartHtml = `
                        <div class="mb-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Knowledge Base Requests</h3>
                       
                            <div class="bg-white p-4 rounded-lg shadow-sm border border-gray-200">
                                <svg width="${width}" height="${height}" class="overflow-visible">
                                    <!-- Y-axis grid lines and labels -->
                                    ${yTicks.map(tick => `
                                        <line x1="${margin.left}" y1="${margin.top + chartHeight - (tick.value / maxValue) * chartHeight}"
                                              x2="${width - margin.right}" y2="${margin.top + chartHeight - (tick.value / maxValue) * chartHeight}"
                                              stroke="#e5e7eb" stroke-width="1" />
                                        <text x="${margin.left - 10}" y="${margin.top + chartHeight - (tick.value / maxValue) * chartHeight + 4}"
                                              font-size="12" fill="#6b7280" text-anchor="end">${tick.label}</text>
                                    `).join('')}
                                    
                                    <!-- X-axis -->
                                    <line x1="${margin.left}" y1="${margin.top + chartHeight}"
                                          x2="${width - margin.right}" y2="${margin.top + chartHeight}"
                                          stroke="#6b7280" stroke-width="1" />
                                    
                                    <!-- X-axis labels - full month names, horizontal -->
                                    ${monthLabels.map(label => {
                                        const x = margin.left + label.index * (barWidth + barPadding) + barPadding / 2 + barWidth / 2;
                                        const y = margin.top + chartHeight + 32;
                                        return `
                                            <text x="${x}" y="${y}"
                                                  font-size="11" fill="#6b7280" text-anchor="middle">${label.display}</text>
                                        `;
                                    }).join('')}
                                    
                                    <!-- Bars -->
                                    ${bars.map(bar => `
                                        <rect x="${bar.x}" y="${bar.y}" width="${bar.width}" height="${bar.height}"
                                              fill="${bar.isLatest ? '#3b82f6' : '#93c5fd'}"
                                              class="bar"
                                              data-month="${bar.month}"
                                              data-value="${bar.value}"
                                              style="cursor: pointer;">
                                            <title>${bar.month}: ${bar.value} requests</title>
                                        </rect>
                                    `).join('')}
                                    
                                    <!-- Y-axis label -->
                                    <text x="15" y="${margin.top + chartHeight / 2}"
                                          font-size="14" fill="#6b7280" text-anchor="middle"
                                          transform="rotate(-90, 15, ${margin.top + chartHeight / 2})">Requests</text>
                                    
                                    <!-- X-axis label -->
   
                                </svg>
                                
                                <!-- Tooltip -->
                            </div>
                           
                        </div>
                  
                    `;
                } else {
                    chartHtml = `
                        <div class="mb-6">
                            <h3 class="text-lg font-semibold text-gray-900 mb-4">Website Crawl Requests</h3>
                            <div class="bg-blue-50 p-4 rounded-lg mb-6">
                                <div class="flex justify-between items-center">
                                    <div>
                                        <h4 class="font-medium text-gray-900">Latest Month Requests</h4>
                                        <p class="text-3xl font-bold text-blue-600 mt-2">0</p>
                                    </div>
                                    <div class="text-right">
                                        <div class="text-sm text-gray-500">Latest Month</div>
                                        <div class="text-lg font-medium text-gray-900">N/A</div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                            <p class="text-gray-500">No crawl requests data available.</p>
                        </div>
                    `;
                }
                
                // Only update the content if it's different to prevent flickering
                if (crawlRequestsContent.innerHTML !== chartHtml) {
                    crawlRequestsContent.innerHTML = chartHtml;
                }
                
                // Add interactivity to the chart after rendering
                if (months.length > 0) {
                    const bars = document.querySelectorAll('.bar');
                    const tooltip = document.getElementById('crawl-chart-tooltip');
                    
                    // Helper function to format month names
                    const formatMonth = (monthStr) => {
                        const months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
                        const parts = monthStr.split('-');
                        if (parts.length === 2) {
                            const year = parts[0].slice(2); // Get last 2 digits of year
                            const monthIndex = parseInt(parts[1], 10) - 1;
                            return months[monthIndex] + " '" + year;
                        }
                        return monthStr;
                    };
                    
                    bars.forEach(bar => {
                        bar.addEventListener('mouseenter', (e) => {
                            const month = e.target.getAttribute('data-month');
                            const value = e.target.getAttribute('data-value');
                            
                            tooltip.innerHTML = `
                                <div class="font-medium">${formatMonth(month)}</div>
                                <div class="text-blue-300">${value} requests</div>
                            `;
                            tooltip.classList.remove('hidden');
                            
                            // Position tooltip
                            const rect = e.target.getBoundingClientRect();
                            tooltip.style.left = (rect.left + rect.width / 2 - tooltip.offsetWidth / 2) + 'px';
                            tooltip.style.top = (rect.top - tooltip.offsetHeight - 10) + 'px';
                        });
                        
                        bar.addEventListener('mouseleave', () => {
                            tooltip.classList.add('hidden');
                        });
                    });
                }
            },

            handleChangeEmail: async function(e) {
                e.preventDefault();

                const newEmail = document.getElementById('new-email').value;
                const confirmEmail = document.getElementById('confirm-email').value;
                const currentPassword = document.getElementById('current-password-email').value;

                // Validate inputs
                if (newEmail !== confirmEmail) {
                    this.showAccountStatus('Email addresses do not match', 'error');
                    return;
                }
                if (!this.isValidEmail(newEmail)) {
                    this.showAccountStatus('Please enter a valid email address', 'error');
                    return;
                }

                try {
                    this.showAccountStatus('Updating email address...', 'info');

                    if (!window.firebaseAuth || !window.firebaseEmailAuthProvider || !window.firebaseReauthenticateWithCredential || !window.firebaseUpdateEmail) {
                        throw new Error('Authentication not initialized');
                    }

                    const user = window.firebaseAuth.currentUser;
                    if (!user || !user.email) {
                        this.showAccountStatus('You must be signed in to update your email.', 'error');
                        return;
                    }

                    // Reauthenticate with current password
                    const credential = window.firebaseEmailAuthProvider.credential(user.email, currentPassword);
                    await window.firebaseReauthenticateWithCredential(user, credential);

                    // Update email in Firebase Auth
                    await window.firebaseUpdateEmail(user, newEmail);

                    // Update cached and UI state
                    if (this.currentUser) this.currentUser.email = newEmail;
                    this.loadCurrentAccountInfo();

                    document.getElementById('change-email-form').reset();
                    this.showAccountStatus('Email address updated successfully!', 'success');
                } catch (error) {
                    console.error('Error updating email:', error);
                    let msg = 'Failed to update email address. Please try again.';
                    if (error.code === 'auth/wrong-password') msg = 'Current password is incorrect.';
                    if (error.code === 'auth/invalid-email') msg = 'Invalid email format.';
                    if (error.code === 'auth/email-already-in-use') msg = 'This email is already in use.';
                    if (error.code === 'auth/requires-recent-login') msg = 'Please sign in again and retry.';
                    this.showAccountStatus(msg, 'error');
                }
            },

            
            handleChangePasswordAuth: async function(e) {
                e.preventDefault();

                const currentPassword = document.getElementById('current-password').value;
                const newPassword = document.getElementById('new-password').value;
                const confirmPassword = document.getElementById('confirm-password').value;

                // Validate inputs
                if (newPassword !== confirmPassword) {
                    this.showAccountStatus('New passwords do not match', 'error');
                    return;
                }

                if (newPassword.length < 8) {
                    this.showAccountStatus('Password must be at least 8 characters long', 'error');
                    return;
                }

                try {
                    this.showAccountStatus('Updating password...', 'info');

                    if (!window.firebaseAuth || !window.firebaseEmailAuthProvider || !window.firebaseReauthenticateWithCredential || !window.firebaseUpdatePassword) {
                        throw new Error('Authentication not initialized');
                    }

                    const user = window.firebaseAuth.currentUser;
                    if (!user || !user.email) {
                        this.showAccountStatus('You must be signed in to update your password.', 'error');
                        return;
                    }

                    // Reauthenticate with current password
                    const credential = window.firebaseEmailAuthProvider.credential(user.email, currentPassword);
                    await window.firebaseReauthenticateWithCredential(user, credential);

                    // Update password in Firebase Auth
                    await window.firebaseUpdatePassword(user, newPassword);

                    // Clear form
                    document.getElementById('change-password-form').reset();

                    this.showAccountStatus('Password updated successfully!', 'success');

                } catch (error) {
                    console.error('Error updating password:', error);
                    let msg = 'Failed to update password. Please try again.';
                    if (error.code === 'auth/wrong-password') msg = 'Current password is incorrect.';
                    if (error.code === 'auth/weak-password') msg = 'Password is too weak. Please choose a stronger password.';
                    if (error.code === 'auth/requires-recent-login') msg = 'Please sign in again and retry.';
                    this.showAccountStatus(msg, 'error');
                }
            },
            showAccountStatus: function(message, type) {
                const statusDiv = document.getElementById('account-status');
                statusDiv.className = `p-4 rounded-lg mb-4 ${type === 'success' ? 'bg-green-100 text-green-800' :
                                                                type === 'error' ? 'bg-red-100 text-red-800' :
                                                                'bg-blue-100 text-blue-800'}`;
                statusDiv.textContent = message;
                statusDiv.classList.remove('hidden');

                // Auto-hide after 5 seconds for success/error messages
                if (type === 'success' || type === 'error') {
                    setTimeout(() => {
                        statusDiv.classList.add('hidden');
                    }, 5000);
                }
            },

            isValidEmail: function(email) {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                return emailRegex.test(email);
            },



            initializeAITraining: function() {
                // Initialize Firestore
                if (window.firebaseFirestore) {
                    try {
                        this.firestore = window.firebaseFirestore;
                        this.firestoreDoc = window.firebaseDoc;
                        this.firestoreSetDoc = window.firebaseSetDoc;
                        this.firestoreGetDoc = window.firebaseGetDoc;
                        this.firestoreCollection = window.firebaseCollection;
        //                console.log('Firestore initialized for AI Training');
                    } catch (error) {
                        console.error('Failed to initialize Firestore:', error);
                    }
                }

                // Setup AI Training event listeners
                this.setupAITrainingListeners();

                // Initialize range sliders
                this.initializeRangeSliders();
            },

            setupAITrainingListeners: function() {
                // Form submission
                const form = document.getElementById('ai-training-form');
                if (form) {
                    form.addEventListener('submit', (e) => this.saveAIConfiguration(e));
                }

                // Button listeners
                const loadConfigBtn = document.getElementById('load-current-config');
                if (loadConfigBtn) {
                    loadConfigBtn.addEventListener('click', () => this.loadCurrentAIConfig());
                }

                const resetBtn = document.getElementById('reset-to-defaults');
                if (resetBtn) {
                    resetBtn.addEventListener('click', () => this.resetToDefaults());
                }

                const testBtn = document.getElementById('test-configuration');
                if (testBtn) {
                    testBtn.addEventListener('click', () => this.testConfiguration());
                }

                const refreshBtn = document.getElementById('refresh-config');
                if (refreshBtn) {
                    refreshBtn.addEventListener('click', () => this.loadCurrentAIConfig());
                }
            },

            initializeRangeSliders: function() {
                // Temperature slider
                const tempSlider = document.getElementById('sampling-temperature');
                const tempValue = document.getElementById('temperature-value');
                if (tempSlider && tempValue) {
                    tempSlider.addEventListener('input', (e) => {
                        tempValue.textContent = e.target.value;
                    });
                }

                // Top P slider
                const topPSlider = document.getElementById('top-p');
                const topPValue = document.getElementById('top-p-value');
                if (topPSlider && topPValue) {
                    topPSlider.addEventListener('input', (e) => {
                        topPValue.textContent = e.target.value;
                    });
                }
            },

            saveAIConfiguration: async function(e) {
                e.preventDefault();

                if (!this.firestore) {
                    this.showTrainingStatus('Firestore not initialized', 'error');
                    return;
                }

                try {
                    // Get form data
                    const formData = new FormData(e.target);
                    const config = {
                        systemMessage: formData.get('systemMessage'),
                        samplingTemperature: parseFloat(formData.get('samplingTemperature')),
                        topK: parseInt(formData.get('topK')),
                        topP: parseFloat(formData.get('topP')),
                        maxTokens: parseInt(formData.get('maxTokens')),
                        updatedAt: new Date(),
                        updatedBy: this.currentUser?.email || 'admin'
                    };

                    // Validate configuration
                    if (!this.validateAIConfig(config)) {
                        return;
                    }

                    // Show loading
                    const saveBtn = document.getElementById('save-training-config');
                    const originalText = saveBtn.textContent;
                    saveBtn.textContent = 'Saving...';
                    saveBtn.disabled = true;

                    // Save to Firestore
                    const docRef = this.firestoreDoc(this.firestore, 'aiTraining', 'config');
                    await this.firestoreSetDoc(docRef, config);

                 

                    this.showTrainingStatus('Configuration saved successfully!', 'success');
                    this.updateCurrentConfigDisplay(config);

                } catch (error) {
                    console.error('Error saving AI configuration:', error);
                    this.showTrainingStatus('Failed to save configuration: ' + error.message, 'error');
                } finally {
                    // Reset button
                    const saveBtn = document.getElementById('save-training-config');
                    saveBtn.textContent = 'Save Configuration';
                    saveBtn.disabled = false;
                }
            },

            loadCurrentAIConfig: async function() {
                if (!this.firestore) {
                    console.warn('Firestore not initialized');
                    return;
                }

                try {
                    const docRef = this.firestoreDoc(this.firestore, 'aiTraining', 'config');
                    const docSnap = await this.firestoreGetDoc(docRef);

                    if (docSnap.exists()) {
                        const config = docSnap.data();
                        this.populateForm(config);
                        this.updateCurrentConfigDisplay(config);

                        // Update last updated time
                        const lastUpdated = document.getElementById('last-updated');
                        if (lastUpdated && config.updatedAt) {
                            const date = config.updatedAt.toDate ? config.updatedAt.toDate() : new Date(config.updatedAt);
                            lastUpdated.textContent = date.toLocaleString();
                        }
                    } else {
                        this.showTrainingStatus('No configuration found. Using defaults.', 'info');
                        this.resetToDefaults();
                    }
                } catch (error) {
                    console.error('Error loading AI configuration:', error);
                    this.showTrainingStatus('Failed to load configuration: ' + error.message, 'error');
                }
            },

            populateForm: function(config) {
                // Populate form fields with loaded configuration
                document.getElementById('system-message').value = config.systemMessage || '';
                document.getElementById('sampling-temperature').value = config.samplingTemperature || 0.7;
                document.getElementById('temperature-value').textContent = config.samplingTemperature || 0.7;
                document.getElementById('top-k').value = config.topK || 40;
                document.getElementById('top-p').value = config.topP || 0.9;
                document.getElementById('top-p-value').textContent = config.topP || 0.9;
                document.getElementById('max-tokens').value = config.maxTokens || 150;
            },

            resetToDefaults: function() {
                // Reset form to default values
                const defaults = {
                    systemMessage: 'You are a helpful AI assistant. Be concise, accurate, and friendly in your responses.',
                    samplingTemperature: 0.7,
                    topK: 40,
                    topP: 0.9,
                    maxTokens: 150
                };

                this.populateForm(defaults);
                this.showTrainingStatus('Form reset to default values', 'info');
            },

            validateAIConfig: function(config) {
                // Validate system message
                if (!config.systemMessage || config.systemMessage.trim().length === 0) {
                    this.showTrainingStatus('System message is required', 'error');
                    return false;
                }

                // Validate temperature
                if (config.samplingTemperature < 0 || config.samplingTemperature > 2) {
                    this.showTrainingStatus('Sampling temperature must be between 0 and 2', 'error');
                    return false;
                }

                // Validate Top K
                if (config.topK < 1 || config.topK > 100) {
                    this.showTrainingStatus('Top K must be between 1 and 100', 'error');
                    return false;
                }

                // Validate Top P
                if (config.topP < 0 || config.topP > 1) {
                    this.showTrainingStatus('Top P must be between 0 and 1', 'error');
                    return false;
                }

                // Validate max tokens
                if (config.maxTokens < 1 || config.maxTokens > 4000) {
                    this.showTrainingStatus('Maximum tokens must be between 1 and 4000', 'error');
                    return false;
                }

                return true;
            },

            showTrainingStatus: function(message, type) {
                const statusDiv = document.getElementById('training-status');
                if (!statusDiv) return;

                statusDiv.className = 'mt-4 p-4 rounded-lg';
                statusDiv.classList.remove('hidden');

                switch (type) {
                    case 'success':
                        statusDiv.classList.add('bg-green-50', 'text-green-700', 'border', 'border-green-200');
                        break;
                    case 'error':
                        statusDiv.classList.add('bg-red-50', 'text-red-700', 'border', 'border-red-200');
                        break;
                    case 'info':
                        statusDiv.classList.add('bg-blue-50', 'text-blue-700', 'border', 'border-blue-200');
                        break;
                    default:
                        statusDiv.classList.add('bg-gray-50', 'text-gray-700', 'border', 'border-gray-200');
                }

                statusDiv.textContent = message;

                // Auto-hide after 5 seconds
                setTimeout(() => {
                    statusDiv.classList.add('hidden');
                }, 5000);
            },

            updateCurrentConfigDisplay: function(config) {
                const displayDiv = document.getElementById('current-config-display');
                if (!displayDiv) return;

                const configHtml = `
                    <div class="space-y-3">
                        <div class="grid grid-cols-2 gap-4">
                            <div>
                                <span class="text-sm font-medium text-gray-600">Temperature:</span>
                                <span class="text-sm text-gray-900 ml-2">${config.samplingTemperature}</span>
                            </div>
                            <div>
                                <span class="text-sm font-medium text-gray-600">Top K:</span>
                                <span class="text-sm text-gray-900 ml-2">${config.topK}</span>
                            </div>
                            <div>
                                <span class="text-sm font-medium text-gray-600">Top P:</span>
                                <span class="text-sm text-gray-900 ml-2">${config.topP}</span>
                            </div>
                            <div>
                                <span class="text-sm font-medium text-gray-600">Max Tokens:</span>
                                <span class="text-sm text-gray-900 ml-2">${config.maxTokens}</span>
                            </div>
                        </div>
                        <div>
                            <span class="text-sm font-medium text-gray-600">System Message:</span>
                            <p class="text-sm text-gray-900 mt-1 p-2 bg-white rounded border">${config.systemMessage}</p>
                        </div>
                    </div>
                `;

                displayDiv.innerHTML = configHtml;
            },

            // AI Training Functions for Main Content Area
            initializeAITrainingMain: function() {
                // Initialize Firestore
                if (window.firebaseFirestore) {
                    try {
                        this.firestore = window.firebaseFirestore;
                        this.firestoreDoc = window.firebaseDoc;
                        this.firestoreSetDoc = window.firebaseSetDoc;
                        this.firestoreGetDoc = window.firebaseGetDoc;

                        console.log('Firestore initialized for AI training');
                    } catch (error) {
                        console.error('Error initializing Firestore:', error);
                        this.showTrainingStatusMain('Error initializing database connection', 'error');
                        return;
                    }
                } else {
                    console.error('Firestore not available');
                    this.showTrainingStatusMain('Database not available', 'error');
                    return;
                }

                // Add event listeners for the main form
                const form = document.getElementById('ai-training-form-main');
                if (form) {
                    form.addEventListener('submit', (e) => {
                        e.preventDefault();
                        this.saveAIConfigurationMain();
                    });
                }

                // Temperature slider
                const tempSlider = document.getElementById('sampling-temperature-main');
                const tempValue = document.getElementById('temperature-value-main');
                if (tempSlider && tempValue) {
                    tempSlider.addEventListener('input', (e) => {
                        tempValue.textContent = e.target.value;
                    });
                }

                const resetBtn = document.getElementById('reset-to-defaults-main');
                if (resetBtn) {
                    resetBtn.addEventListener('click', () => this.resetToDefaultsMain());
                }

                // AI Agent Toggle Button
                const aiAgentToggle = document.getElementById('ai-agent-toggle');
                if (aiAgentToggle) {
                    aiAgentToggle.addEventListener('click', () => this.toggleAIAgent());
                }

                this.aiTrainingInitialized = true;
                console.log('AI Training main interface initialized');
                
                // Load current AI agent status
                this.loadAIAgentStatus();
            },

            saveAIConfigurationMain: async function() {
                if (!this.firestore) {
                    this.showTrainingStatusMain('Database not available', 'error');
                    return;
                }

                try {
                    this.showTrainingStatusMain('Saving configuration...', 'info');

                    // Get form data
                    const config = {
                        systemMessage: document.getElementById('system-message-main').value,
                        samplingTemperature: parseFloat(document.getElementById('sampling-temperature-main').value),
                        updatedAt: new Date().toISOString(),
                        updatedBy: this.currentUser?.email || 'admin'
                    };

                    // Validate configuration
                    if (!config.systemMessage.trim()) {
                        this.showTrainingStatusMain('System message is required', 'error');
                        return;
                    }

                    // Save to Firestore
                    const docRef = this.firestoreDoc(this.firestore, 'aiTraining', 'config');
                    await this.firestoreSetDoc(docRef, config);

                    this.showSuccessPopup('Configuration saved successfully!');
                    
                    // Also save the AI agent status
                    await this.saveAIAgentStatus();
                } catch (error) {
                    console.error('Error saving configuration:', error);
                    this.showTrainingStatusMain('Error saving configuration: ' + error.message, 'error');
                }
            },

            loadCurrentAIConfigMain: async function() {
                if (!this.firestore) {
                    this.showTrainingStatusMain('Database not available', 'error');
                    return;
                }

                try {
                    this.showTrainingStatusMain('Loading configuration...', 'info');

                    const docRef = this.firestoreDoc(this.firestore, 'aiTraining', 'config');
                    const docSnap = await this.firestoreGetDoc(docRef);

                    if (docSnap.exists()) {
                        const config = docSnap.data();

                        // Populate form fields
                        document.getElementById('system-message-main').value = config.systemMessage || '';
                        document.getElementById('sampling-temperature-main').value = config.samplingTemperature || 0.7;
                        document.getElementById('temperature-value-main').textContent = config.samplingTemperature || 0.7;

                        this.showTrainingStatusMain('Configuration loaded successfully!', 'success');
                    } else {
                        this.showTrainingStatusMain('No saved configuration found', 'warning');
                        this.resetToDefaultsMain();
                    }
                } catch (error) {
                    console.error('Error loading configuration:', error);
                    this.showTrainingStatusMain('Error loading configuration: ' + error.message, 'error');
                }
            },

            resetToDefaultsMain: function() {
                document.getElementById('system-message-main').value = 'You are a helpful AI assistant. Be friendly, professional, and provide accurate information.';
                document.getElementById('sampling-temperature-main').value = 0.7;
                document.getElementById('temperature-value-main').textContent = '0.7';

                this.showTrainingStatusMain('Reset to default values', 'info');
            },

            // AI Agent Toggle Functions
            toggleAIAgent: function() {
                const toggle = document.getElementById('ai-agent-toggle');
                const handle = document.getElementById('ai-agent-toggle-handle');
                
                // Toggle the visual state
                if (toggle.classList.contains('bg-gray-200')) {
                    // Currently inactive, activate it
                    toggle.classList.remove('bg-gray-200');
                    toggle.classList.add('bg-blue-600');
                    handle.classList.remove('translate-x-0');
                    handle.classList.add('translate-x-5');
                    toggle.setAttribute('aria-checked', 'true');
                } else {
                    // Currently active, deactivate it
                    toggle.classList.remove('bg-blue-600');
                    toggle.classList.add('bg-gray-200');
                    handle.classList.remove('translate-x-5');
                    handle.classList.add('translate-x-0');
                    toggle.setAttribute('aria-checked', 'false');
                }
            },

            loadAIAgentStatus: async function() {
                if (!this.firestore) {
                    console.warn('Firestore not available for loading AI agent status');
                    return;
                }

                try {
                    const docRef = this.firestoreDoc(this.firestore, 'aiTraining', 'aiActive');
                    const docSnap = await this.firestoreGetDoc(docRef);
                    
                    const toggle = document.getElementById('ai-agent-toggle');
                    const handle = document.getElementById('ai-agent-toggle-handle');
                    
                    if (docSnap.exists()) {
                        const data = docSnap.data();
                        const isActive = data.status === true;
                        
                        // Update the toggle UI to match the loaded status
                        if (isActive) {
                            toggle.classList.remove('bg-gray-200');
                            toggle.classList.add('bg-blue-600');
                            handle.classList.remove('translate-x-0');
                            handle.classList.add('translate-x-5');
                            toggle.setAttribute('aria-checked', 'true');
                        } else {
                            toggle.classList.remove('bg-blue-600');
                            toggle.classList.add('bg-gray-200');
                            handle.classList.remove('translate-x-5');
                            handle.classList.add('translate-x-0');
                            toggle.setAttribute('aria-checked', 'false');
                        }
                    } else {
                        // Default to inactive if no document exists
                        toggle.classList.remove('bg-blue-600');
                        toggle.classList.add('bg-gray-200');
                        handle.classList.remove('translate-x-5');
                        handle.classList.add('translate-x-0');
                        toggle.setAttribute('aria-checked', 'false');
                    }
                } catch (error) {
                    console.error('Error loading AI agent status:', error);
                }
            },

            saveAIAgentStatus: async function() {
                if (!this.firestore) {
                    console.warn('Firestore not available for saving AI agent status');
                    return;
                }

                try {
                    const toggle = document.getElementById('ai-agent-toggle');
                    const isActive = toggle.getAttribute('aria-checked') === 'true';
                    
                    const docRef = this.firestoreDoc(this.firestore, 'aiTraining', 'aiActive');
                    await this.firestoreSetDoc(docRef, {
                        status: isActive,
                        updatedAt: new Date().toISOString(),
                        updatedBy: this.currentUser?.email || 'admin'
                    });
                    
                    console.log('AI agent status saved successfully');
                } catch (error) {
                    console.error('Error saving AI agent status:', error);
                }
            },

            showTrainingStatusMain: function(message, type) {
                const statusDiv = document.getElementById('training-status-main');
                if (!statusDiv) return;

                statusDiv.className = 'mt-4 p-4 rounded-lg';
                statusDiv.classList.remove('hidden');

                switch (type) {
                    case 'success':
                        statusDiv.classList.add('bg-green-100', 'text-green-800', 'border', 'border-green-200');
                        break;
                    case 'error':
                        statusDiv.classList.add('bg-red-100', 'text-red-800', 'border', 'border-red-200');
                        break;
                    case 'warning':
                        statusDiv.classList.add('bg-yellow-100', 'text-yellow-800', 'border', 'border-yellow-200');
                        break;
                    case 'info':
                    default:
                        statusDiv.classList.add('bg-blue-100', 'text-blue-800', 'border', 'border-blue-200');
                        break;
                }

                statusDiv.textContent = message;

                // Auto-hide after 5 seconds for success/info messages
                if (type === 'success' || type === 'info') {
                    setTimeout(() => {
                        statusDiv.classList.add('hidden');
                    }, 5000);
                }
            },

            // Show success popup notification
            showSuccessPopup: function(message) {
                const popup = document.getElementById('success-popup');
                const messageSpan = document.getElementById('popup-message');

                if (popup && messageSpan) {
                    messageSpan.textContent = message;
                    popup.classList.remove('hidden');
                    popup.classList.add('animate-bounce');

                    // Auto-hide after 5 seconds
                    setTimeout(() => {
                        popup.classList.add('opacity-0');
                        setTimeout(() => {
                            popup.classList.add('hidden');
                            popup.classList.remove('opacity-0', 'animate-bounce');
                        }, 300);
                    }, 5000);
                }
            },

            loadknowledgebaseTab: function() {
                // Set up form listener
                this.setupKnowledgeBaseListeners();
                
                // Load current status from Firebase if it exists
                const statusRef = window.firebaseRef(window.firebaseDatabase, 'knowledgebaseTab');
                window.firebaseOnValue(statusRef, (snapshot) => {
                    if (snapshot.exists()) {
                        const statusData = snapshot.val();
                        this.displayStatus(statusData);
                        // Apply restrictions based on status
                        this.applyStatusRestrictions(statusData.status);
                    } else {
                        // If no status exists, ensure form is enabled
                        this.applyStatusRestrictions('idle');
                    }
                });
                
                // Set up listener for discovered URLs
                this.setupDiscoveredUrlsListener();
            },

            setupKnowledgeBaseListeners: function() {
                // Form submission
                const form = document.getElementById('fetch-url-form');
                if (form) {
                    form.addEventListener('submit', (e) => this.handleFetchUrl(e));
                }
            },

            handleFetchUrl: async function(e) {
                e.preventDefault();
                
                const urlInput = document.getElementById('website-url');
                const url = urlInput.value.trim();
                
                if (!url) {
                    alert('Please enter a valid URL');
                    return;
                }
                
                // Check current status before proceeding
                const statusRef = window.firebaseRef(window.firebaseDatabase, 'knowledgebaseTab');
                const statusSnapshot = await new Promise((resolve) => {
                    window.firebaseOnValue(statusRef, (snapshot) => {
                        resolve(snapshot);
                    }, { onlyOnce: true });
                });
                
                // If status is processing, don't allow new requests
          
                
                // Store processing status in Firebase
                await this.storeProcessingStatus(url);
                
                // Send webhook request
                await this.sendWebhookRequest(url);
            },

            sendWebhookRequest: async function(url) {
                try {
                    const response = await fetch(`${window.API_CONFIG.baseUrl}/proxy.php?action=knowledgebase`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json'
                        },
                        body: JSON.stringify({
                            payload: {
                                url: url,
                                status: 'crawl_request_sent'
                            }
                        })
                    });
                    
                    if (!response.ok) {
                        console.error('Webhook request failed:', response.statusText);
                    }
                } catch (error) {
                    console.error('Error sending webhook request:', error);
                }
            },

            storeProcessingStatus: async function(url) {
                try {
                    const statusRef = window.firebaseRef(window.firebaseDatabase, 'knowledgebaseTab');
                    await window.firebaseSet(statusRef, {
                        status: 'processing',
                        url: url,
                        timestamp: Date.now()
                    });
                } catch (error) {
                    console.error('Error storing processing status:', error);
                }
            },

            displayStatus: function(statusData) {
                const statusContainer = document.getElementById('knowledgebase-status-container');
                const statusContent = document.getElementById('knowledgebase-status-content');
                
                if (!statusData) {
                    statusContainer.classList.add('hidden');
                    return;
                }
                
                statusContainer.classList.remove('hidden');
                
                let statusHtml = '';
                if (statusData.status === 'processing') {
                    statusHtml = `
                        <div class="flex items-center">
                            <div class="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-500 mr-3"></div>
                            <div>
                                <p class="font-medium text-gray-900">Processing URL: ${statusData.url}</p>
                                <p class="text-sm text-gray-500">Started at: ${new Date(statusData.timestamp).toLocaleString()}</p>
                            </div>
                        </div>
                    `;
                } else {
                    statusHtml = `
                        <div class="flex items-center">
                            <div class="w-6 h-6 rounded-full bg-green-500 flex items-center justify-center mr-3">
                                <svg class="w-4 h-4 text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M5 13l4 4L19 7"></path>
                                </svg>
                            </div>
                            <div>
                                <p class="font-medium text-gray-900">Processed URL: ${statusData.url}</p>
                                <p class="text-sm text-gray-500">Completed at: ${new Date(statusData.timestamp).toLocaleString()}</p>
                            </div>
                        </div>
                    `;
                }
                
                statusContent.innerHTML = statusHtml;
            },

applyStatusRestrictions: function(status) {
const urlInput = document.getElementById('website-url');
const fetchDataBtn = document.getElementById('fetch-data-btn');

if (status === 'processing') {
    // Disable form elements when processing
    urlInput.disabled = true;
    fetchDataBtn.disabled = true;
    fetchDataBtn.textContent = 'Processing...';
} else {
    // Enable form elements when not processing
    urlInput.disabled = false;
    fetchDataBtn.disabled = false;
    fetchDataBtn.textContent = 'Fetch Data';
}
},

            // Set up listener for discovered URLs
            setupDiscoveredUrlsListener: function() {
                const urlsRef = window.firebaseRef(window.firebaseDatabase, 'knowledgebaseTab/discoveredURLs');
                window.firebaseOnValue(urlsRef, (snapshot) => {
                    if (snapshot.exists()) {
                        const urlsData = snapshot.val();
                        this.displayDiscoveredUrls(urlsData);
                    } else {
                        // No URLs found, display empty state
                        this.displayDiscoveredUrls(null);
                    }
                }, (error) => {
                    console.error('Error listening to discovered URLs:', error);
                    this.displayDiscoveredUrls(null, error);
                });
            },

            // Display discovered URLs as clickable hyperlinks
            displayDiscoveredUrls: function(urlsData, error = null) {
                const urlsContainer = document.getElementById('discovered-urls-container');
                const urlsContent = document.getElementById('discovered-urls-content');
                
                // Handle error state
                if (error) {
                    urlsContent.innerHTML = `
                        <div class="text-red-500 text-sm">
                            Error loading URLs: ${error.message || 'Unknown error'}
                        </div>
                    `;
                    urlsContainer.classList.remove('hidden');
                    return;
                }
                
                // Handle empty state
                if (!urlsData || Object.keys(urlsData).length === 0) {
                    urlsContent.innerHTML = `
                        <div class="text-gray-500 text-sm">
                            No URLs discovered yet
                        </div>
                    `;
                    urlsContainer.classList.remove('hidden');
                    return;
                }
                
                // Display URLs
                let urlsHtml = '';
                Object.keys(urlsData).forEach(key => {
                    const url = urlsData[key];
                    // Validate URL format
                    try {
                        new URL(url);
                        urlsHtml += `
                            <div class="flex items-center p-2 bg-gray-50 rounded hover:bg-gray-100">
                                <a href="${url}" target="_blank" rel="noopener noreferrer"
                                   class="text-blue-600 hover:text-blue-800 hover:underline truncate flex-1">
                                    ${url}
                                </a>
                                <svg class="w-4 h-4 text-gray-400 ml-2 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 6H6a2 2 0 00-2 2v10a2 2 0 002 2h10a2 2 0 002-2v-4M14 4h6m0 0v6m0-6L10 14"></path>
                                </svg>
                            </div>
                        `;
                    } catch (e) {
                        // Handle invalid URLs
                        urlsHtml += `
                            <div class="flex items-center p-2 bg-red-50 rounded">
                                <span class="text-red-600 truncate flex-1">
                                    Invalid URL: ${url}
                                </span>
                            </div>
                        `;
                    }
                });
                
                urlsContent.innerHTML = urlsHtml;
                urlsContainer.classList.remove('hidden');
            }
        };

        // Initialize when DOM is loaded
        document.addEventListener('DOMContentLoaded', function () {
            // Wait for Firebase to be available
            const checkFirebase = setInterval(() => {
                if (window.firebaseDatabase && window.firebaseFirestore) {
                    clearInterval(checkFirebase);
                    window.AdminCRM.init();
                }
            }, 100);
        });

        // Expose debug functions globally
        window.debugUpdateCounters = function() {
            if (window.AdminCRM) {
                window.AdminCRM.forceUpdateCounters();
            } else {
                console.error('AdminCRM not initialized');
            }
        };

        // All admin credential management is now handled securely through Firebase Authentication